import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/core/utils/url_container.dart';
import 'package:viserpay/core/utils/util.dart';
import 'package:viserpay/data/controller/nominee/nominee_controller.dart';
import 'package:viserpay/data/controller/savings/savings_controller.dart';
import 'package:viserpay/data/repo/nominee/nominee_repo.dart';
import 'package:viserpay/data/repo/saving/saving_repo.dart';
import 'package:viserpay/data/services/api_service.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/card/account_details_card.dart';
import 'package:viserpay/view/components/card/cash_2colum.dart';
import 'package:viserpay/view/components/cash-card/title_card.dart';
import 'package:viserpay/view/components/cash-card/user/user_card.dart';
import 'package:viserpay/view/components/column_widget/card_column.dart';
import 'package:viserpay/view/components/dialog/app_dialog.dart';
import 'package:viserpay/view/components/image/my_image_widget.dart';
import 'package:viserpay/view/components/snack_bar/show_custom_snackbar.dart';
import 'package:viserpay/view/components/text-form-field/customPinText.dart';
import 'package:viserpay/view/screens/savings/widget/dps_bank_widget.dart';

class SavingsPinScreen extends StatefulWidget {
  const SavingsPinScreen({super.key});

  @override
  State<SavingsPinScreen> createState() => _SavingsPinScreenState();
}

class _SavingsPinScreenState extends State<SavingsPinScreen> {
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(SavingRepo(apiClient: Get.find()));
    Get.put(NomineeRepo(apiClient: Get.find()));
    Get.put(NomineeController(nomineeRepo: Get.find()));
    final controller = Get.put(SavingsController(savingRepo: Get.find(), nomineeController: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.clearPinData();
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: CustomAppBar(title: MyStrings.savings, isTitleCenter: true),
      body: GetBuilder<SavingsController>(
        builder: (controller) {
          return SingleChildScrollView(
            padding: Dimensions.defaultPaddingHV,
            child: GetBuilder<NomineeController>(
              builder: (nomineeController) {
                return Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    // DpsBankWidget(dps: controller.selectedDpsPlan!),
                    Container(
                      padding: EdgeInsets.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space10),
                      decoration: BoxDecoration(
                        color: MyColor.colorWhite,
                        borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                        border: Border.all(color: MyColor.borderColor, width: 1),
                        boxShadow: MyUtils.getCardShadow(),
                      ),
                      child: Column(
                        children: [
                          Row(
                            mainAxisAlignment: MainAxisAlignment.spaceBetween,
                            crossAxisAlignment: CrossAxisAlignment.center,
                            children: [
                              Expanded(
                                child: Row(
                                  mainAxisAlignment: MainAxisAlignment.start,
                                  crossAxisAlignment: CrossAxisAlignment.center,
                                  children: [
                                    MyImageWidget(
                                      imageUrl: "${UrlContainer.domainUrl}/${controller.organizationImagePath}/${controller.selectedOrganization?.image}",
                                      radius: 30,
                                      height: 60,
                                      width: 60,
                                    ),
                                    const SizedBox(width: Dimensions.space10),
                                    Expanded(
                                      child: Text(controller.selectedOrganization?.name ?? "", style: semiBoldDefault.copyWith()),
                                    ),
                                  ],
                                ),
                              ),
                            ],
                          ),
                          const SizedBox(height: Dimensions.space10),
                          Row(
                            mainAxisAlignment: MainAxisAlignment.spaceBetween,
                            children: [
                              CardColumn(
                                header: MyStrings.installment,
                                body: "",
                                bodyWidget: Text.rich(
                                  TextSpan(
                                    text: "${controller.currencySym}${StringConverter.formatNumber(controller.selectedDpsPlan?.installmentAmount ?? "", precision: 0)}/",
                                    style: lightMediumLarge.copyWith(color: MyColor.getGreyText()),
                                    children: [
                                      TextSpan(text: "${controller.selectedInterval?.name}", style: regularDefault.copyWith(fontSize: 10)),
                                    ],
                                  ),
                                ),
                                headerTextStyle: boldDefault.copyWith(fontSize: Dimensions.fontLarge - 1, fontWeight: FontWeight.w500),
                                space: 5,
                              ),
                              CardColumn(
                                header: MyStrings.interest,
                                body: "${StringConverter.formatNumber(controller.selectedDpsPlan?.interestRate ?? "", precision: 2)}%",
                                headerTextStyle: boldDefault.copyWith(fontSize: Dimensions.fontLarge - 1, fontWeight: FontWeight.w500),
                                bodyTextStyle: lightMediumLarge.copyWith(color: MyColor.getGreyText()),
                              ),
                              CardColumn(
                                header: MyStrings.tenure,
                                body: controller.selectedTenure?.name ?? "",
                                headerTextStyle: boldDefault.copyWith(fontSize: Dimensions.fontLarge - 1, fontWeight: FontWeight.w500),
                                bodyTextStyle: lightMediumLarge.copyWith(color: MyColor.getGreyText()),
                                alignmentEnd: true,
                              ),
                            ],
                          ),
                          const SizedBox(height: Dimensions.space10),
                        ],
                      ),
                    ),

                    const SizedBox(height: Dimensions.space16),
                    AccountDetailsCard(
                      amount: "${controller.currencySym}${StringConverter.formatNumber(controller.selectedDpsPlan?.installmentAmount ?? "")}",
                      charge: "${controller.currencySym}${StringConverter.formatNumber(controller.selectedDpsPlan?.maturityAmount ?? "")}",
                      total: controller.selectedTenure?.name ?? "",
                      totalTitle: MyStrings.duration,
                      chargeTitle: MyStrings.maturityAmount,
                      amountTitle: MyStrings.installment,
                    ),
                    const SizedBox(height: Dimensions.space20),
                    if (controller.otpType.isNotEmpty) ...[
                      Text(MyStrings.selectOtpType.tr, style: mediumDefault.copyWith()),
                      SingleChildScrollView(
                        physics: const BouncingScrollPhysics(),
                        scrollDirection: Axis.horizontal,
                        padding: EdgeInsets.zero,
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.start,
                          children: List.generate(
                            controller.otpType.length,
                            (index) => Row(
                              mainAxisAlignment: MainAxisAlignment.spaceBetween,
                              children: [
                                Row(
                                  children: [
                                    Checkbox(
                                      value: controller.selectedOtpType == controller.otpType[index] ? true : false,
                                      onChanged: (p) {
                                        controller.selectOtpType(controller.otpType[index]);
                                      },
                                      shape: const CircleBorder(),
                                      activeColor: MyColor.primaryDark,
                                    ),
                                    GestureDetector(
                                      onTap: () {
                                        controller.selectOtpType(controller.otpType[index]);
                                      },
                                      child: Text(
                                        controller.otpType[index].toUpperCase(),
                                        style: semiBoldDefault.copyWith(
                                          color: controller.selectedOtpType?.toLowerCase() == controller.otpType[index].toLowerCase() ? MyColor.colorBlack : MyColor.primaryDark,
                                        ),
                                      ),
                                    ),
                                  ],
                                ),
                              ],
                            ),
                          ),
                        ),
                      ),
                    ],

                    CustomPinField(
                      onChanged: (p) {
                        MyUtils.vibrate();
                      },
                      controller: controller.pinController,
                      focusNode: controller.pinFocusNode,
                      needOutlineBorder: true,
                      labelText: "",
                      hintText: MyStrings.enterYourPIN,
                      isShowSuffixIcon: true,
                      textInputType: TextInputType.number,
                      inputAction: TextInputAction.done,
                      prefixIcon: const SizedBox(
                        width: 22,
                        height: 22,
                        child: Padding(
                          padding: EdgeInsets.all(8.0),
                          child: Icon(
                            Icons.lock,
                            color: MyColor.primaryColor,
                          ),
                        ),
                      ),
                      suffixWidget: GestureDetector(
                        onTap: () {
                          bool isBalanceAvailable = StringConverter.biggerThen(controller.currentBalance, controller.selectedDpsPlan?.installmentAmount ?? "");
                          if (!isBalanceAvailable) {
                            CustomSnackBar.error(errorList: [MyStrings.savingsEmptyInstallmentMsg]);
                            return;
                          }
                          String newBalance = StringConverter.minus(controller.currentBalance, controller.selectedDpsPlan?.installmentAmount ?? "");

                          if (controller.otpType.isEmpty) {
                            if (controller.validatePinCode()) {
                              submitDialog(context, controller, newBalance, nomineeController);
                            }
                          } else {
                            if (controller.validatePinCode() == true) {
                              if (controller.selectedOtpType == 'null') {
                                CustomSnackBar.error(errorList: [MyStrings.pleaseSelectOtp.tr]);
                              } else {
                                submitDialog(context, controller, newBalance, nomineeController);
                              }
                            }
                          }
                        },
                        child: const SizedBox(
                          width: 22,
                          height: 22,
                          child: Padding(padding: EdgeInsets.all(5.0), child: Icon(Icons.arrow_right_alt_sharp, color: MyColor.primaryColor)),
                        ),
                      ),
                      onSubmit: () {
                        bool isBalanceAvailable = StringConverter.biggerThen(controller.currentBalance, controller.selectedDpsPlan?.installmentAmount ?? "");
                        if (!isBalanceAvailable) {
                          CustomSnackBar.error(errorList: [MyStrings.savingsEmptyInstallmentMsg]);
                          return;
                        }
                        String newBalance = StringConverter.minus(controller.currentBalance, controller.selectedDpsPlan?.installmentAmount ?? "");

                        if (controller.otpType.isEmpty) {
                          if (controller.validatePinCode()) {
                            submitDialog(context, controller, newBalance, nomineeController);
                          }
                        } else {
                          if (controller.validatePinCode() == true) {
                            if (controller.selectedOtpType == 'null') {
                              CustomSnackBar.error(errorList: [MyStrings.pleaseSelectOtp.tr]);
                            } else {
                              submitDialog(context, controller, newBalance, nomineeController);
                            }
                          }
                        }
                      },
                    ),
                  ],
                );
              },
            ),
          );
        },
      ),
    );
  }

  void submitDialog(BuildContext context, SavingsController controller, String newBalance, NomineeController nomineeController) {
    AppDialog().confirmDialog(
      context,
      title: MyStrings.savings.tr,
      userDetails: DpsBankWidget(dps: controller.selectedDpsPlan!),
      cashDetails: CashDetailsColumn(
        total: "${controller.currencySym}${StringConverter.formatNumber(controller.selectedDpsPlan?.installmentAmount ?? "")}",
        newBalance: "${controller.currencySym}${StringConverter.formatNumber(newBalance)}",
        charge: "${controller.currencySym}0.0",
        hideCharge: true,
        firstTitle: MyStrings.installmentAmount,
      ),
      onFinish: () {},
      onWaiting: () {
        controller.submitDpsPlan();
      },
    );
  }
}
